<?php
/**
 * Plugin support: Elementor
 *
 * @package WordPress
 * @subpackage ThemeREX Addons
 * @since v1.0
 */

// Don't load directly
if ( ! defined( 'TRX_ADDONS_VERSION' ) ) {
	die( '-1' );
}


// Check if plugin 'Elementor' is installed and activated
// Attention! This function is used in many files and was moved to the api.php
/*
if ( !function_exists( 'trx_addons_exists_elementor' ) ) {
	function trx_addons_exists_elementor() {
		return class_exists('Elementor\Plugin');
	}
}
*/

// Return true if Elementor exists and current mode is preview
if ( !function_exists( 'trx_addons_elm_is_preview' ) ) {
	function trx_addons_elm_is_preview() {
		return trx_addons_exists_elementor() 
				&& (\Elementor\Plugin::instance()->preview->is_preview_mode()
					|| (trx_addons_get_value_gp('post') > 0
						&& trx_addons_get_value_gp('action') == 'elementor'
						)
					);
	}
}

	
// Merge specific styles into single stylesheet
if ( !function_exists( 'trx_addons_elm_merge_styles' ) ) {
	add_filter("trx_addons_filter_merge_styles", 'trx_addons_elm_merge_styles');
	function trx_addons_elm_merge_styles($list) {
		if (trx_addons_exists_elementor()) {
			$list[] = TRX_ADDONS_PLUGIN_API . 'elementor/_elementor.scss';
		}
		return $list;
	}
}


// Merge shortcode's specific styles to the single stylesheet (responsive)
if ( !function_exists( 'trx_addons_elm_merge_styles_responsive' ) ) {
	add_filter("trx_addons_filter_merge_styles_responsive", 'trx_addons_elm_merge_styles_responsive');
	function trx_addons_elm_merge_styles_responsive($list) {
		if (trx_addons_exists_elementor()) {
			$list[] = TRX_ADDONS_PLUGIN_API . 'elementor/_elementor.responsive.scss';
		}
		return $list;
	}
}

	
// Merge plugin's specific scripts into single file
if ( !function_exists( 'trx_addons_elm_merge_scripts' ) ) {
	add_action("trx_addons_filter_merge_scripts", 'trx_addons_elm_merge_scripts');
	function trx_addons_elm_merge_scripts($list) {
		if (trx_addons_exists_elementor()) {
			$list[] = TRX_ADDONS_PLUGIN_API . 'elementor/elementor.js';
			$list[] = TRX_ADDONS_PLUGIN_API . 'elementor/elementor-parallax.js';
		}
		return $list;
	}
}


// Add responsive sizes
if ( !function_exists( 'trx_addons_elm_sass_responsive' ) ) {
	add_filter("trx_addons_filter_sass_responsive", 'trx_addons_elm_sass_responsive', 11);
	function trx_addons_elm_sass_responsive($list) {
		if (!isset($list['md_lg']))
			$list['md_lg'] = array(
									'min' => $list['sm']['max']+1,
									'max' => $list['lg']['max']
									);
		return $list;
	}
}

// Load required styles and scripts for Elementor Editor mode
if ( !function_exists( 'trx_addons_elm_editor_load_scripts' ) ) {
	add_action("elementor/editor/before_enqueue_scripts", 'trx_addons_elm_editor_load_scripts');
	function trx_addons_elm_editor_load_scripts() {
		trx_addons_load_scripts_admin(true);
		trx_addons_localize_scripts_admin();
		wp_enqueue_style(  'trx_addons-elementor-editor', trx_addons_get_file_url(TRX_ADDONS_PLUGIN_API . 'elementor/elementor.editor.css'), array(), null );
		wp_enqueue_script( 'trx_addons-elementor-editor', trx_addons_get_file_url(TRX_ADDONS_PLUGIN_API . 'elementor/elementor.editor.js'), array('jquery'), null, true );
		do_action('trx_addons_action_pagebuilder_admin_scripts');
	}
}

// Add vars to the Elementors editor
if ( !function_exists( 'trx_addons_elm_localize_admin_scripts' ) ) {
	add_filter( 'trx_addons_filter_localize_script_admin',	'trx_addons_elm_localize_admin_scripts');
	function trx_addons_elm_localize_admin_scripts($vars = array()) {
		$vars['add_hide_on_xxx'] = trx_addons_get_setting('add_hide_on_xxx');
		return $vars;
	}
}

// Load required scripts for Elementor Preview mode
if ( !function_exists( 'trx_addons_elm_preview_load_scripts' ) ) {
	add_action("elementor/frontend/after_enqueue_scripts", 'trx_addons_elm_preview_load_scripts');
	function trx_addons_elm_preview_load_scripts() {
		if ( trx_addons_is_on(trx_addons_get_option('debug_mode')) ) {
			wp_enqueue_script( 'trx_addons-elementor-preview', trx_addons_get_file_url(TRX_ADDONS_PLUGIN_API . 'elementor/elementor.js'), array('jquery'), null, true );
			wp_enqueue_script( 'trx_addons-elementor-parallax', trx_addons_get_file_url(TRX_ADDONS_PLUGIN_API . 'elementor/elementor-parallax.js'), array('jquery'), null, true );
		}
		wp_enqueue_script( 'tweenmax', trx_addons_get_file_url('js/tweenmax/tweenmax.min.js'), array(), null, true );
		if ( trx_addons_elm_is_preview() ) {
			do_action('trx_addons_action_pagebuilder_preview_scripts', 'elementor');
		}
	}
}
	
// Add shortcode's specific vars into JS storage
if ( !function_exists( 'trx_addons_elm_localize_script' ) ) {
	add_filter("trx_addons_filter_localize_script", 'trx_addons_elm_localize_script');
	function trx_addons_elm_localize_script($vars) {
		$vars['elementor_stretched_section_container'] = get_option('elementor_stretched_section_container');
		return $vars;
	}
}

// Return true if specified post/page is built with Elementor
if ( !function_exists( 'trx_addons_is_built_with_elementor' ) ) {
	function trx_addons_is_built_with_elementor( $post_id ) {
		// Elementor\DB::is_built_with_elementor` is soft deprecated since 3.2.0
		// Use `Plugin::$instance->documents->get( $post_id )->is_built_with_elementor()` instead
		// return trx_addons_exists_elementor() && \Elementor\Plugin::instance()->db->is_built_with_elementor( $post_id );
		$rez = false;
		if ( trx_addons_exists_elementor() && ! empty( $post_id ) ) {
			$document = \Elementor\Plugin::instance()->documents->get( $post_id );
			if ( is_object( $document ) ) {
				$rez = $document->is_built_with_elementor();
			}
		}
		return $rez;
	}
}

if ( ! function_exists( 'trx_addons_elm_is_experiment_active' ) ) {
	/**
	 * Check if the Elementor experiment is active
	 *
	 * @param string $experiment  Experiment name
	 * 
	 * @return bool  true if the experiment is active
	 */
	function trx_addons_elm_is_experiment_active( $experiment ) {
		return trx_addons_exists_elementor()
				&& ! empty( \Elementor\Plugin::instance()->experiments )
				&& is_object( \Elementor\Plugin::instance()->experiments )
				&& method_exists( \Elementor\Plugin::instance()->experiments, 'is_feature_active' )
				&& \Elementor\Plugin::instance()->experiments->is_feature_active( $experiment );
	}
}

if ( ! function_exists( 'trx_addons_deregister_swiper_from_elementor' ) ) {
	add_action( 'wp_enqueue_scripts', 'trx_addons_deregister_swiper_from_elementor', 9999 );
	/**
	 * Deregister Swiper from Elementor to use the plugin's version if a plugin setting 'replace_swiper_from_elementor' is enabled
	 * 
	 * @hooked wp_enqueue_scripts
	 * 
	 * @trigger trx_addons_filter_replace_swiper_from_elementor
	 */
	function trx_addons_deregister_swiper_from_elementor() {
		if ( trx_addons_exists_elementor() ) {
			$replace = apply_filters( 'trx_addons_filter_replace_swiper_from_elementor', trx_addons_get_setting( 'replace_swiper_from_elementor', 'none' ) );
			if ( $replace == 'always' || ( $replace == 'old' && ! trx_addons_elm_is_experiment_active( 'e_swiper_latest' ) ) ) {
				if ( wp_script_is( 'swiper', 'registered' ) ) {
					// Deregister Swiper from Elementor
					wp_deregister_script( 'swiper' );
					if ( wp_style_is( 'swiper', 'registered' ) ) {
						wp_deregister_style( 'swiper' );
					}
					// Enqueue Swiper from the plugin
					trx_addons_enqueue_slider( 'swiper', true );
				}
			}
		}
	}
}

// Return url with post edit link
if ( !function_exists( 'trx_addons_elm_post_edit_link' ) ) {
	add_filter( 'trx_addons_filter_post_edit_link', 'trx_addons_elm_post_edit_link', 10, 2 );
	function trx_addons_elm_post_edit_link( $link, $post_id ) {
		if ( trx_addons_exists_elementor() && trx_addons_is_built_with_elementor( $post_id ) ) {
			$link = str_replace( 'action=edit', 'action=elementor', $link );
		}
		return $link;
	}
}

// Change "Go Pro" links
//----------------------------------------------
if (!function_exists('trx_addons_elm_change_gopro_plugins') && defined('ELEMENTOR_PLUGIN_BASE')) {
	add_filter( 'plugin_action_links_' . ELEMENTOR_PLUGIN_BASE, 'trx_addons_elm_change_gopro_plugins', 11 );
	function trx_addons_elm_change_gopro_plugins($links) {
		if (!empty($links['go_pro']) && preg_match('/href="([^"]*)"/', $links['go_pro'], $matches) && !empty($matches[1])) {
			$links['go_pro'] = str_replace($matches[1], trx_addons_add_to_url($matches[1], array('ref' => '2496')), $links['go_pro']);
		}
		return $links;
	}
}
if (!function_exists('trx_addons_elm_change_gopro_dashboard')) {
	add_filter( 'elementor/admin/dashboard_overview_widget/footer_actions', 'trx_addons_elm_change_gopro_dashboard', 11 );
	function trx_addons_elm_change_gopro_dashboard($actions) {
		if (!empty($actions['go-pro']['link'])) {
			$actions['go-pro']['link'] = trx_addons_add_to_url($actions['go-pro']['link'], array('ref' => '2496'));
		}
		return $actions;
	}
}
if (!function_exists('trx_addons_elm_change_gopro_menu')) {
	add_filter( 'wp_redirect', 'trx_addons_elm_change_gopro_menu', 11, 2 );
	function trx_addons_elm_change_gopro_menu($link, $status=0) {
		if (strpos($link, '//elementor.com/pro/') !== false) {
			$link = trx_addons_add_to_url($link, array('ref' => '2496'));
		}
		return $link;
	}
}


// Init Elementor's support
//--------------------------------------------------------

// Set Elementor's options at once
if (!function_exists('trx_addons_elm_init_once')) {
	add_action( 'init', 'trx_addons_elm_init_once', 2 );
	function trx_addons_elm_init_once() {
		if (trx_addons_exists_elementor() && !get_option('trx_addons_setup_elementor_options', false)) {
			// Set components specific values to the Elementor's options
			do_action('trx_addons_action_set_elementor_options');
			// Set flag to prevent change Elementor's options again
			update_option('trx_addons_setup_elementor_options', 1);
		}
	}
}

// Add categories for widgets, shortcodes, etc.
if (!function_exists('trx_addons_elm_add_categories')) {
	add_action( 'elementor/elements/categories_registered', 'trx_addons_elm_add_categories' );
	function trx_addons_elm_add_categories($mgr = null) {

		static $added = false;

		if (!$added) {

			if ($mgr == null) $mgr = \Elementor\Plugin::instance()->elements_manager;
			
			// Add a custom category for ThemeREX Addons Shortcodes
			$mgr->add_category( 
				'trx_addons-elements',
				array(
					'title' => __( 'ThemeREX Addons Elements', 'trx_addons' ),
					'icon' => 'eicon-apps', //default icon
					'active' => true,
				)
			);

			// Add a custom category for ThemeREX Addons Widgets
			$mgr->add_category( 
				'trx_addons-widgets',
				array(
					'title' => __( 'ThemeREX Addons Widgets', 'trx_addons' ),
					'icon' => 'eicon-gallery-grid', //default icon
					'active' => false,
				)
			);

			// Add a custom category for ThemeREX Addons CPT
			$mgr->add_category( 
				'trx_addons-cpt',
				array(
					'title' => __( 'ThemeREX Addons Extensions', 'trx_addons' ),
					'icon' => 'eicon-gallery-grid', //default icon
					'active' => false,
				)
			);

			// Add a custom category for third-party shortcodes
			$mgr->add_category( 
				'trx_addons-support',
				array(
					'title' => __( 'ThemeREX Addons Support', 'trx_addons' ),
					'icon' => 'eicon-woocommerce', //default icon
					'active' => false,
				)
			);

			$added = true;
		}
	}
}

// Template to create our classes with widgets
//---------------------------------------------
if (!function_exists('trx_addons_elm_init')) {
	add_action( 'elementor/init', 'trx_addons_elm_init' );
	function trx_addons_elm_init() {

		// Add categories (for old Elementor)
		trx_addons_elm_add_categories();

		// Define class for our shortcodes and widgets
		require_once TRX_ADDONS_PLUGIN_DIR . TRX_ADDONS_PLUGIN_API . 'elementor/elementor-class-widget.php';
	}
}


// Check if icon name is from the Elementor icons
if ( !function_exists( 'trx_addons_is_elementor_icon' ) ) {
	function trx_addons_is_elementor_icon($icon) {
		return !empty($icon) && strpos($icon, 'fa ') !== false;
	}
}



// Output inline CSS
// if current action is 'wp_ajax_elementor_render_widget' or 'admin_action_elementor' (old Elementor) or 'elementor_ajax' (new Elementor)
// (called from Elementor Editor via AJAX or first load page content to the Editor)
//---------------------------------------------------------------------------------------
if (!function_exists('trx_addons_elm_print_inline_css')) {
	add_filter( 'elementor/widget/render_content', 'trx_addons_elm_print_inline_css', 10, 2 );
	function trx_addons_elm_print_inline_css($content, $widget=null) {
		if (doing_action('wp_ajax_elementor_render_widget') || doing_action('admin_action_elementor') || doing_action('elementor_ajax') || doing_action('wp_ajax_elementor_ajax')) {
			$css = trx_addons_get_inline_css(true);
			if (!empty($css)) {
				$content .= sprintf('<style type="text/css">%s</style>', $css);
			}
		}
		return $content;
	}
}

// Return an action for controls registration (changed in v.3.5.0)
if ( ! function_exists( 'trx_addons_elementor_get_action_for_controls_registration' ) ) {
	function trx_addons_elementor_get_action_for_controls_registration() {
		return defined( 'ELEMENTOR_VERSION' ) && version_compare( ELEMENTOR_VERSION, '3.5.0', '>=' )
					? 'elementor/controls/register'
					: 'elementor/controls/controls_registered';
	}
}



// Fix for the Elementor Pro widget wrappers
if ( !function_exists( 'trx_addons_elm_pro_woocommerce_wordpress_widget_css_class' ) ) {
	add_filter( 'elementor/widgets/wordpress/widget_args', 'trx_addons_elm_pro_woocommerce_wordpress_widget_css_class', 11, 2 );
	function trx_addons_elm_pro_woocommerce_wordpress_widget_css_class( $default_widget_args, $widget ) {
		if ( is_object( $widget ) ) {
			$widget_instance = $widget->get_widget_instance();
			if ( ! empty( $widget_instance->widget_cssclass ) ) {
				$open_tag = sprintf( '<div class="%s">', $widget_instance->widget_cssclass );
				if ( substr( $default_widget_args['before_widget'], -strlen( $open_tag ) ) == $open_tag
					&& $default_widget_args['after_widget'] == '</aside></div>'
				) {
					$default_widget_args['after_widget'] = '</div></aside>';
				}
			}
		}
		return $default_widget_args;
	}
}


// Wrapper for Elementor widgets registration to prevent a deprecation warning
if ( !function_exists( 'trx_addons_elm_register_widget' ) ) {
	function trx_addons_elm_register_widget( $widget_class ) {
		if ( class_exists( $widget_class ) ) {
			// Plugin::$instance->widgets_manager->register_widget_type() is soft deprecated since 3.5.0
			// Use Plugin::$instance->widgets_manager->register() instead
			if ( method_exists( \Elementor\Plugin::instance()->widgets_manager, 'register' ) ) {
				\Elementor\Plugin::instance()->widgets_manager->register( new $widget_class() );
			} else {
				\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new $widget_class() );
			}
		}
	}
}



// Register custom controls for Elementor
//------------------------------------------------------------------------
if (!function_exists('trx_addons_elm_register_custom_controls')) {
	add_action( trx_addons_elementor_get_action_for_controls_registration(), 'trx_addons_elm_register_custom_controls' );
	function trx_addons_elm_register_custom_controls( $controls_manager ) {
		$controls = array('trx_icons');
		foreach ( $controls as $control_id ) {
			$control_filename = str_replace('_', '-', $control_id);
			require_once TRX_ADDONS_PLUGIN_DIR . TRX_ADDONS_PLUGIN_API . "elementor/params/{$control_filename}/{$control_filename}.php";
			$class_name = 'Trx_Addons_Elementor_Control_' . ucwords( $control_id );
			if ( defined( 'ELEMENTOR_VERSION' ) && version_compare( ELEMENTOR_VERSION, '3.5.0', '>=' ) ) {
				$controls_manager->register( new $class_name() );	
			} else {
				$controls_manager->register_control( $control_id, new $class_name() );
			}
		}
	}
}


// Add/Modify/Remove standard Elementor's shortcodes params
//------------------------------------------------------------------------

// Add/Remove shortcodes params to the existings sections
if (!function_exists('trx_addons_elm_add_params_inside_section')) {
	add_action( 'elementor/element/before_section_end', 'trx_addons_elm_add_params_inside_section', 10, 3 );
	function trx_addons_elm_add_params_inside_section($element, $section_id, $args) {

		if (!is_object($element)) return;
		
		$el_name = $element->get_name();
		
		// Add 'Hide bg image on XXX' to the rows
		if ( ($el_name == 'section' && $section_id == 'section_background')
			|| ($el_name == 'column' && $section_id == 'section_style')
			) {

			$element->add_control( 'hide_bg_image_on_tablet', array(
									'type' => \Elementor\Controls_Manager::SWITCHER,
									'label' => __( 'Hide bg image on the tablet', 'trx_addons' ),
									'label_on' => __( 'Hide', 'trx_addons' ),
									'label_off' => __( 'Show', 'trx_addons' ),
									'return_value' => 'tablet',
									'prefix_class' => 'hide_bg_image_on_',
								) );
			$element->add_control( 'hide_bg_image_on_mobile', array(
									'type' => \Elementor\Controls_Manager::SWITCHER,
									'label' => __( 'Hide bg image on the mobile', 'trx_addons' ),
									'label_on' => __( 'Hide', 'trx_addons' ),
									'label_off' => __( 'Show', 'trx_addons' ),
									'return_value' => 'mobile',
									'prefix_class' => 'hide_bg_image_on_',
								) );
		}

		// Add 'Hide on XXX' to the any elements
		$add_hide_on_xxx = trx_addons_get_setting('add_hide_on_xxx');
        if ( ! trx_addons_is_off($add_hide_on_xxx) && class_exists( 'TRX_Addons_Elementor_Widget' ) ) {
			if ($section_id == '_section_responsive') { // && $el_name == 'section'
				$params = TRX_Addons_Elementor_Widget::get_hide_param(false);
				if (is_array($params)) {
					if ($add_hide_on_xxx == 'add') {
						$element->add_control(
							'trx_addons_responsive_heading',
							array(
								'label' => __( 'Theme-specific params', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::HEADING,
								'separator' => 'before',
							)
						);
						$element->add_control(
							'trx_addons_responsive_description',
							array(
								'raw' => __( "Theme-specific parameters - you can use them instead of the Elementor's parameters above.", 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::RAW_HTML,
								'content_classes' => 'elementor-descriptor',
							)
						);
					}
					foreach ($params as $p) {
						$element->add_control( $p['name'], array_merge($p, array(
																				'return_value' => $p['name'],
																				'prefix_class' => 'sc_layouts_',
																				))
											);
					}
				}
			}
		}

		// Add 'Extend background' and 'Background mask' to the rows, columns and text-editor
		if ( ($el_name == 'section' && $section_id == 'section_background')
			|| ($el_name == 'column' && $section_id == 'section_style')
			|| ($el_name == 'text-editor' && $section_id == 'section_background')
			) {
			$element->add_control( 'extra_bg', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Extend background", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_extra_bg(''),
									'default' => '',
									'prefix_class' => 'sc_extra_bg_'
									) );
			$element->add_control( 'extra_bg_mask', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Background mask", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_extra_bg_mask(''),
									'default' => '',
									'prefix_class' => 'sc_bg_mask_'
									) );
		}

		// Add 'Alter height/gap' to the spacer and divider
		if ( ($el_name == 'spacer' && $section_id == 'section_spacer')
				  || ($el_name == 'divider' && $section_id == 'section_divider')) {
			$element->add_control( 'alter_height', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => $el_name == 'divider' ? __("Alter gap", 'trx_addons') : __("Alter height", 'trx_addons'),
									'label_block' => true,
									'options' => trx_addons_get_list_sc_empty_space_heights(''),
									'default' => '',
									'prefix_class' => 'sc_height_'
									) );
		}
		
		// Add new shapes to the 'Shape dividers' in the section
		global $TRX_ADDONS_STORAGE;
		if ( $el_name == 'section' && $section_id == 'section_shape_divider' && !empty($TRX_ADDONS_STORAGE['shapes_list'])) {
			$sides = array('top', 'bottom');
			$options = $conditions = false;
			$prefix = 'trx_addons';
			foreach ($sides as $side) {
				// Add shapes to the shapes list
				$control_id = "shape_divider_{$side}";
				if ($options === false) {
					$control = $element->get_controls( $control_id );
					if ( ! empty( $controls['options'] ) && is_array( $controls['options'] ) ) {
						$options = $control['options'];
						foreach ($TRX_ADDONS_STORAGE['shapes_list'] as $shape) {
							$shape_name = pathinfo($shape, PATHINFO_FILENAME);
							$options["{$prefix}_{$shape_name}"] = ucfirst(str_replace('_', ' ', $shape_name));
						}
					}
				}
				$element->update_control( $control_id, array(
									'options' => $options
								) );

				// Add shapes to the condition for the 'Flip' and 'Width' controls
				$controls = array("flip", "width");
				$conditions = array();
				foreach ($controls as $control_name) {
					$control_id = "shape_divider_{$side}_{$control_name}";
					$control = $element->get_controls( $control_id );
					$conditions[$control_name] = isset($control['condition']) ? $control['condition'] : false;
					if (is_array($conditions[$control_name])) {
						foreach($TRX_ADDONS_STORAGE['shapes_list'] as $shape) {
							$shape_name = pathinfo($shape, PATHINFO_FILENAME);
							foreach ($conditions[$control_name] as $k=>$v) {
								if (is_array($v) && strpos($k, 'shape_divider_')!==false) {
									$v[] = "{$prefix}_{$shape_name}";
									$conditions[$control_name][$k] = $v;
								}
							}
						}
					}
				}
				foreach ($controls as $control_name) {
					$control_id = "shape_divider_{$side}_{$control_name}";
					if ($conditions[$control_name] !== false) {
						$element->update_control( $control_id, array(
										'condition' => $conditions[$control_name]
									) );
					}
				}
			}
		}
	}
}


// Add/Remove column params to the new section
if (!function_exists('trx_addons_elm_add_params_to_columns')) {
	add_action( 'elementor/element/after_section_end', 'trx_addons_elm_add_params_to_columns', 10, 3 );
	function trx_addons_elm_add_params_to_columns($element, $section_id, $args) {

		if ( !is_object($element) ) return;
		
		if ( ($element->get_name() == 'section' && $section_id == 'section_layout')
				||
			 ($element->get_name() == 'column' && $section_id == 'layout')
		) {
			
			$element->start_controls_section( 'section_trx_layout',	array(
																		'tab' => \Elementor\Controls_Manager::TAB_LAYOUT,
																		'label' => __( 'Position', 'trx_addons' )
																	) );
			// Add 'Fix column' to the columns
			if ($element->get_name() == 'column') {
				$element->add_control( 'fix_column', array(
									'type' => \Elementor\Controls_Manager::SWITCHER,
									'label' => __( 'Fix column', 'trx_addons' ),
									'description' => wp_kses_data( __("Fix this column when page scrolling. Attention! At least one column in the row must have a greater height than this column", 'trx_addons') ),
									'label_on' => __( 'Fix', 'trx_addons' ),
									'label_off' => __( 'No', 'trx_addons' ),
									'return_value' => 'fixed',
									'prefix_class' => 'sc_column_',
									) );
			}
			$element->add_control( 'shift_x', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Shift block along the X-axis", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_shift(''),
									'default' => '',
									'prefix_class' => 'sc_shift_x_'
									) );
			$element->add_control( 'shift_y', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Shift block along the Y-axis", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_shift(''),
									'default' => '',
									'prefix_class' => 'sc_shift_y_'
									) );
			
			$element->add_control( 'push_x', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Push block along the X-axis", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_shift(''),
									'default' => '',
									'prefix_class' => 'sc_push_x_'
									) );
			$element->add_control( 'push_y', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Push block along the Y-axis", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_shift(''),
									'default' => '',
									'prefix_class' => 'sc_push_y_'
									) );
			
			$element->add_control( 'pull_x', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Pull next block along the X-axis", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_shift(''),
									'default' => '',
									'prefix_class' => 'sc_pull_x_'
									) );
			$element->add_control( 'pull_y', array(
									'type' => \Elementor\Controls_Manager::SELECT,
									'label' => __("Pull next block along the Y-axis", 'trx_addons'),
									'options' => trx_addons_get_list_sc_content_shift(''),
									'default' => '',
									'prefix_class' => 'sc_pull_y_'
									) );

			$element->end_controls_section();
		}
	}
}


// Add "Parallax" params to the new section to rows
if (!function_exists('trx_addons_elm_add_parallax_params_to_rows')) {
	add_action( 'elementor/element/after_section_end', 'trx_addons_elm_add_parallax_params_to_rows', 10, 3 );
	function trx_addons_elm_add_parallax_params_to_rows($element, $section_id, $args) {

		if ( !is_object($element) ) return;

		if ( $element->get_name() == 'section' && $section_id == 'section_border' ) {

			$element->start_controls_section( 'section_trx_parallax',	array(
																		'tab' => !empty($args['tab']) ? $args['tab'] : \Elementor\Controls_Manager::TAB_ADVANCED,
																		'label' => __( 'Parallax', 'trx_addons' )
																	) );

			$element->add_control(
				'parallax_blocks',
				array(
					'label' => __( 'Parallax blocks', 'trx_addons' ),
					'type' => \Elementor\Controls_Manager::REPEATER,
					'fields' => apply_filters('trx_addons_sc_param_group_params',
						array(
							array(
								'name' => 'type',
								'label' => __( 'Block handle', 'trx_addons' ),
								'label_block' => false,
								'type' => \Elementor\Controls_Manager::SELECT,
								'options' => array(
									'none'   => __('None', 'trx_addons'),
									'mouse'  => __('Mouse events', 'trx_addons'),
									'scroll' => __('Scroll events', 'trx_addons'),
								),
								'default' => 'none',
							),
							array(
								'name' => 'animation_prop',
								'label' => __( 'Animation', 'trx_addons' ),
								'label_block' => false,
								'type' => \Elementor\Controls_Manager::SELECT,
								'options' => array(
									'background'  => __('Background', 'trx_addons'),
									'transform'   => __('Transform', 'trx_addons'),
									'transform3d' => __('Transform3D', 'trx_addons'),
								),
								'default' => 'background',
							),
							array(
								'name' => 'image',
								'label' => __( 'Background image', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::MEDIA,
								'default' => array(
									'url' => '',
								),
							),
							array(
								'name' => 'bg_size',
								'label' => __( 'Background size', 'trx_addons' ),
								'label_block' => false,
								'type' => \Elementor\Controls_Manager::SELECT,
								'options' => array(
									'auto'    => __('Auto', 'trx_addons'),
									'cover'   => __('Cover', 'trx_addons'),
									'contain' => __('Contain', 'trx_addons'),
								),
								'default' => 'background',
							),
							array(
								'name' => 'left',
								'label' => __( 'Left position (in %)', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::SLIDER,
								'default' => array(
									'size' => 0,
									'unit' => 'px'
								),
								'range' => array(
									'px' => array(
										'min' => 0,
										'max' => 100
									),
								),
								'size_units' => array( 'px' )
							),
							array(
								'name' => 'top',
								'label' => __( 'Top position (in %)', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::SLIDER,
								'default' => array(
									'size' => 0,
									'unit' => 'px'
								),
								'range' => array(
									'px' => array(
										'min' => 0,
										'max' => 100
									),
								),
								'size_units' => array( 'px' ),
							),
							array(
								'name' => 'speed',
								'label' => __( 'Shift speed', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::SLIDER,
								'default' => array(
									'size' => 50,
									'unit' => 'px'
								),
								'range' => array(
									'px' => array(
										'min' => 0,
										'max' => 100
									),
								),
								'size_units' => array( 'px' ),
							),
							array(
								'name' => 'z_index',
								'label' => __( 'Z-index', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::SLIDER,
								'default' => array(
									'size' => '',
									'unit' => 'px'
								),
								'range' => array(
									'px' => array(
										'min' => -1,
										'max' => 100
									),
								),
								'size_units' => array( 'px' ),
							),
							array(
								'name' => 'class',
								'label' => __( 'CSS class', 'trx_addons' ),
								'description' => __( 'Class name to assign additional rules to this block. For example: "hide_on_notebook", "hide_on_tablet", "hide_on_mobile" to hide block on the relative device', 'trx_addons' ),
								'type' => \Elementor\Controls_Manager::TEXT,
								'default' => '',
							),
						),
						'trx_sc_parallax_row'),
					'title_field' => '{{{ left.size }}}x{{{ top.size }}} / {{{ type }}} / {{{ animation_prop }}}',
				)
			);

			$element->end_controls_section();
		}
	}
}

// Add "data-parallax-params" to the wrapper of the row
if ( !function_exists( 'trx_addons_elm_add_parallax_data_to_rows' ) ) {
	// Before Elementor 2.1.0
	add_action( 'elementor/frontend/element/before_render',  'trx_addons_elm_add_parallax_data_to_rows', 10, 1 );
	// After Elementor 2.1.0
	add_action( 'elementor/frontend/section/before_render', 'trx_addons_elm_add_parallax_data_to_rows', 10, 1 );
	function trx_addons_elm_add_parallax_data_to_rows($element) {
		if ( is_object($element) && $element->get_name() == 'section' ) {
			$settings = $element->get_settings();
			if (!empty($settings['parallax_blocks']) && is_array($settings['parallax_blocks']) && count($settings['parallax_blocks']) > 0) {
				$element->add_render_attribute( '_wrapper', 'class', 'sc_parallax' );
				$element->add_render_attribute( '_wrapper', 'data-parallax-blocks', json_encode($settings['parallax_blocks']) );
			}
		}
	}
}


// Add "Parallax" params to the new section to widgets
if (!function_exists('trx_addons_elm_add_parallax_params_to_widgets')) {
	add_action( 'elementor/element/after_section_end', 'trx_addons_elm_add_parallax_params_to_widgets', 10, 3 );
	add_action( 'elementor/widget/after_section_end', 'trx_addons_elm_add_parallax_params_to_widgets', 10, 3 );
	function trx_addons_elm_add_parallax_params_to_widgets($element, $section_id, $args) {

		if ( !is_object($element) ) return;

		if ( $element->get_name() == 'common' && in_array($section_id, array('section_border', '_section_border')) ) {
			
			$element->start_controls_section( 'section_trx_parallax', array(
																		'tab' => !empty($args['tab']) ? $args['tab'] : \Elementor\Controls_Manager::TAB_ADVANCED,
																		'label' => __( 'Parallax', 'trx_addons' )
																	) );
			$element->add_control( 'parallax', array(
													'type' => \Elementor\Controls_Manager::SWITCHER,
													'label' => __( 'Parallax', 'trx_addons' ),
													'label_on' => __( 'On', 'trx_addons' ),
													'label_off' => __( 'Off', 'trx_addons' ),
													'return_value' => 'parallax',
													'prefix_class' => 'sc_',
									) );
			$element->add_control( 'parallax_start', array(
													'type' => \Elementor\Controls_Manager::SWITCHER,
													'label' => __( 'Values below are', 'trx_addons' ),
													'label_on' => __( 'Start', 'trx_addons' ),
													'label_off' => __( 'End', 'trx_addons' ),
													'return_value' => 'start',
													'prefix_class' => 'sc_parallax_',
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );
			$element->add_control( 'parallax_text', array(
													'type' => \Elementor\Controls_Manager::SELECT,
													'label' => __( 'Text animation', 'trx_addons' ),
													'label_block' => false,
													'options' => array(
														'block' => __( 'Whole block', 'trx_addons'),
														'words' => __( 'Word by word', 'trx_addons'),
														'chars' => __( 'Char by char', 'trx_addons'),
													),
													'default' => 'block',
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );
			$element->add_control( 'parallax_x', array(
													'label' => __( 'The shift along the X-axis (in px)', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => 0,
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => -500,
															'max' => 500
														)
													),
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );
			$element->add_control( 'parallax_y', array(
													'label' => __( 'The shift along the Y-axis (in px)', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => 0,
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => -500,
															'max' => 500
														)
													),
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );

			$element->add_control( 'parallax_opacity', array(
													'label' => __( 'Change the opacity', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => 0,
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => -1,
															'max' => 0,
															'step' => 0.05
														)
													),
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );

			$element->add_control( 'parallax_scale', array(
													'label' => __( 'Change the scale (in %)', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => 0,
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => -100,
															'max' => 1000,
														)
													),
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );

			$element->add_control( 'parallax_rotate', array(
													'label' => __( 'Change the rotation (in deg)', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => 0,
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => -360,
															'max' => 360,
															'step' => 1
														)
													),
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );

			$element->add_control( 'parallax_time', array(
													'label' => __( 'Duration (in sec)', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => 1,
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => 0.1,
															'max' => 10,
															'step' => 0.1
														)
													),
													'condition' => array(
														'parallax' => 'parallax'
													),
									) );

			$element->end_controls_section();
		}
	}
}

// Add "data-parallax-params" to the wrapper of the widget
if ( !function_exists( 'trx_addons_elm_add_parallax_data_to_widgets' ) ) {
	// Before Elementor 2.1.0
	add_action( 'elementor/frontend/element/before_render',  'trx_addons_elm_add_parallax_data_to_widgets', 10, 1 );
	// After Elementor 2.1.0
	add_action( 'elementor/frontend/widget/before_render',  'trx_addons_elm_add_parallax_data_to_widgets', 10, 1 );
	function trx_addons_elm_add_parallax_data_to_widgets($element) {
		$settings = $element->get_settings();
		if ( ! empty($settings['parallax']) ) {
			$element->add_render_attribute( '_wrapper', 'data-parallax-params', json_encode(array(
				'x' => !empty($settings['parallax_x']) ? $settings['parallax_x']['size'] : 0,
				'y' => !empty($settings['parallax_y']) ? $settings['parallax_y']['size'] : 0,
				'scale' => !empty($settings['parallax_scale']) ? $settings['parallax_scale']['size'] : 0,
				'rotate' => !empty($settings['parallax_rotate']) ? $settings['parallax_rotate']['size'] : 0,
				'opacity' => !empty($settings['parallax_opacity']) ? $settings['parallax_opacity']['size'] : 0,
				'duration' => !empty($settings['parallax_duration']) ? $settings['parallax_duration']['size'] : 1,
				'text' => !empty($settings['parallax_text']) ? $settings['parallax_text'] : 'block',
			)) );
		}
	}
}


// Add "Fly" params to the new section to widgets
if (!function_exists('trx_addons_elm_add_params_fly')) {
	add_action( 'elementor/element/after_section_end', 'trx_addons_elm_add_params_fly', 10, 3 );
	add_action( 'elementor/widget/after_section_end', 'trx_addons_elm_add_params_fly', 10, 3 );
	function trx_addons_elm_add_params_fly($element, $section_id, $args) {

		if ( !is_object($element) ) return;

		if ( in_array($element->get_name(), array('common')) && in_array($section_id, array('section_border', '_section_border')) ) {
			
			$element->start_controls_section( 'section_trx_fly', array(
																		'tab' => !empty($args['tab']) ? $args['tab'] : \Elementor\Controls_Manager::TAB_ADVANCED,
																		'label' => __( 'Fly', 'trx_addons' )
																	) );
			$element->add_control(
				'fly',
				array(
					'label' => __( 'Fly', 'trx_addons' ),
					'label_block' => false,
					'type' => \Elementor\Controls_Manager::SELECT,
					'options' => array_merge(
									array('static' => __('Static', 'trx_addons')),
									array('custom' => __('Custom', 'trx_addons')),
									trx_addons_get_list_sc_positions()
								),
					'default' => 'static',
					'prefix_class' => 'sc_fly_',
				)
			);
			$coord = array(
							'label' => __( 'Left', 'trx_addons' ),
							'type' => \Elementor\Controls_Manager::SLIDER,
							'default' => array(
								'size' => '',
								'unit' => 'px'
							),
							'size_units' => array( 'px', 'em', '%' ),
							'range' => array(
								'px' => array(
									'min' => -500,
									'max' => 500
								),
								'em' => array(
									'min' => -50,
									'max' => 50
								),
								'%' => array(
									'min' => -100,
									'max' => 100
								)
							),
							'condition' => array(
								'fly' => array( 'custom', 'tl', 'tr', 'bl', 'br' )
							),
							'selectors' => array(
								'{{WRAPPER}}' => 'left: {{SIZE}}{{UNIT}};'
							),
						);
			$element->add_responsive_control( 'fly_left', $coord );
			$coord['label'] = __( 'Right', 'trx_addons' );
			$coord['selectors'] = array( '{{WRAPPER}}' => 'right: {{SIZE}}{{UNIT}};' );
			$element->add_responsive_control( 'fly_right', $coord );
			$coord['label'] = __( 'Top', 'trx_addons' );
			$coord['selectors'] = array( '{{WRAPPER}}' => 'top: {{SIZE}}{{UNIT}};' );
			$element->add_responsive_control( 'fly_top', $coord );
			$coord['label'] = __( 'Bottom', 'trx_addons' );
			$coord['selectors'] = array( '{{WRAPPER}}' => 'bottom: {{SIZE}}{{UNIT}};' );
			$element->add_responsive_control( 'fly_bottom', $coord );

			$element->add_responsive_control( 'fly_scale', array(
													'label' => __( 'Scale', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => '',
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => 0,
															'max' => 10,
															'step' => 0.1
														)
													),
													'selectors' => array(
														'{{WRAPPER}} .elementor-widget-container' => 'transform: scale({{SIZE}}, {{SIZE}});'
													),
									) );

			$element->add_responsive_control( 'fly_rotate', array(
													'label' => __( 'Rotation (in deg)', 'trx_addons' ),
													'type' => \Elementor\Controls_Manager::SLIDER,
													'default' => array(
														'size' => '',
														'unit' => 'px'
													),
													'size_units' => array( 'px' ),
													'range' => array(
														'px' => array(
															'min' => -360,
															'max' => 360,
															'step' => 1
														)
													),
													'selectors' => array(
														'{{WRAPPER}} .elementor-widget-container' => 'transform: rotate({{SIZE}}deg);'
													),
									) );

			$element->end_controls_section();
		}
	}
}

// Add class "sc_fly" to the wrapper of widgets
if ( !function_exists( 'trx_addons_elm_render_fly' ) ) {
	// Before Elementor 2.1.0
	add_action( 'elementor/frontend/element/before_render',  'trx_addons_elm_render_fly', 10, 1 );
	// After Elementor 2.1.0
	add_action( 'elementor/frontend/section/before_render', 'trx_addons_elm_render_fly', 10, 1 );
	add_action( 'elementor/frontend/column/before_render',  'trx_addons_elm_render_fly', 10, 1 );
	add_action( 'elementor/frontend/widget/before_render',  'trx_addons_elm_render_fly', 10, 1 );
	function trx_addons_elm_render_fly($element) {
		$settings = $element->get_settings();
		if (!empty($settings['fly'])) {
			//$element->add_render_attribute( '_wrapper', 'class', 'sc_fly');
		}
	}
}


// Substitute shapes in the sections
if (!function_exists('trx_addons_elm_before_render')) {
	// Before Elementor 2.1.0
	add_action( 'elementor/frontend/element/before_render', 'trx_addons_elm_before_render', 10, 1 );
	// After Elementor 2.1.0
	add_action( 'elementor/frontend/section/before_render', 'trx_addons_elm_before_render', 10, 1 );
	function trx_addons_elm_before_render($element) {
		if ( is_object($element) ) {
			$el_name = $element->get_name();
			if ( $el_name == 'section' ) {
				$settings = $element->get_settings();
				$sides = array('top', 'bottom');
				$capture = false;
				$prefix = 'trx_addons';
				foreach ($sides as $side) {
					$base_setting_key = "shape_divider_{$side}";
					$shape = $settings[ $base_setting_key ];
					if (strpos($shape, "{$prefix}_") === 0) {
						$capture = true;
						$shapes = \Elementor\Shapes::get_shapes();
						if (!is_array($shapes)) $shapes = array('mountains'=>'');
						$element->set_settings("{$base_setting_key}_{$prefix}", str_replace("{$prefix}_", '', $shape));
						$element->set_settings($base_setting_key, trx_addons_array_get_first($shapes));
						if (!empty($element->active_settings[$base_setting_key])) {
							$element->active_settings[$base_setting_key] = trx_addons_array_get_first($shapes);
						}
					}
				}
				if ($capture) {
					ob_start();
				}
			}
		}
	}
}

if (!function_exists('trx_addons_elm_after_render')) {
	// Before Elementor 2.1.0
	add_action( 'elementor/frontend/element/after_render', 'trx_addons_elm_after_render', 10, 1 );
	// After Elementor 2.1.0
	add_action( 'elementor/frontend/section/after_render', 'trx_addons_elm_after_render', 10, 1 );
	function trx_addons_elm_after_render($element) {
		if ( is_object($element) ) {
			$el_name = $element->get_name();
			if ( $el_name == 'section' ) {
				$settings = $element->get_settings();
				$sides = array('top', 'bottom');
				$replace = array();
				$prefix = 'trx_addons';
				foreach ($sides as $side) {
					$base_setting_key = "shape_divider_{$side}";
					if (!empty($settings[ "{$base_setting_key}_{$prefix}" ])) {
						$replace["elementor-shape-{$side}"] = $settings[ "{$base_setting_key}_{$prefix}" ];
					}
				}
				if (count($replace) > 0) {
					$html = ob_get_contents();
					ob_end_clean();
					foreach ($replace as $class=>$shape) {
						$shape_dir = trx_addons_get_file_dir("css/shapes/{$shape}.svg");
						if (!empty($shape_dir)) {
							$html = preg_replace('~(<div[\s]*class="elementor-shape[\s]+'.$class.'".*>)([\s\S]*)(</div>)~U',
												'$1' . strip_tags(trx_addons_fgc($shape_dir), '<svg><path>') . '$3',
												$html);
						}
					}
					trx_addons_show_layout($html);
				}
			}
		}
	}
}


// Replace widget's args with theme-specific args
if ( !function_exists( 'trx_addons_elm_wordpress_widget_args' ) ) {
	add_filter( 'elementor/widgets/wordpress/widget_args', 'trx_addons_elm_wordpress_widget_args', 10, 2 );
	function trx_addons_elm_wordpress_widget_args($widget_args, $widget) {
		return trx_addons_prepare_widgets_args($widget->get_name(), $widget->get_name(), $widget_args);
	}
}

// Prepare group atts for the new Elementor version: make associative array from list by key 'name'
if ( !function_exists( 'trx_addons_elm_prepare_group_params' ) ) {
    add_filter( 'trx_addons_sc_param_group_params', 'trx_addons_elm_prepare_group_params', 999 );
    function trx_addons_elm_prepare_group_params( $args ) {
        if ( is_array( $args ) && ! empty( $args[0]['name'] ) ) {
            $new = array();
            foreach( $args as $item ) {
                if ( isset( $item['name'] ) ) {
                    $new[ $item['name'] ] = $item;
                }
            }
            $args = $new;
        }
        return $args;
    }
}

// Move paddings from .elementor-element-wrap to .elementor-column-wrap
// to compatibility with old themes
if ( ! function_exists( 'trx_addons_elm_move_paddings_to_column_wrap' ) ) {
    add_action( 'elementor/element/before_section_end', 'trx_addons_elm_move_paddings_to_column_wrap', 10, 3 );
    function trx_addons_elm_move_paddings_to_column_wrap( $element, $section_id, $args ) {
        if ( is_object( $element ) ) {
            $el_name = $element->get_name();
            // Add one more classname to the selector for paddings of columns
            // to override theme-specific rules
            if ( 'column' == $el_name && 'section_advanced' == $section_id ) {
                $element->update_responsive_control( 'padding', array(
                                            'selectors' => array(
                                                '{{WRAPPER}} > .elementor-element-populated.elementor-column-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',    // Elm 2.9- (or DOM Optimization == Inactive)
                                                '{{WRAPPER}} > .elementor-element-populated.elementor-widget-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',    // Elm 3.0+
                                            )
                                        ) );
            }
        }
    }
}

// Return true if Elementor exists and current mode is edit
if ( !function_exists( 'trx_addons_elm_is_edit_mode' ) ) {
	function trx_addons_elm_is_edit_mode() {
		static $is_edit_mode = -1;
		if ( $is_edit_mode === -1 ) {
			$is_edit_mode = trx_addons_exists_elementor()
								&& ( \Elementor\Plugin::instance()->editor->is_edit_mode()
									|| ( trx_addons_get_value_gp( 'post' ) > 0
										&& trx_addons_get_value_gp( 'action' ) == 'elementor'
										)
									|| ( is_admin()
										&& in_array( trx_addons_get_value_gp( 'action' ), array( 'elementor', 'elementor_ajax', 'wp_ajax_elementor_ajax' ) )
										)
									);
		}
		return $is_edit_mode;
	}
}

// Elementor 3.21.0+ - add default value for the 'spacer' widget
//--------------------------------------------------
if ( ! function_exists( 'trx_addons_elm_add_spacer_default_value' ) ) {
	add_action( 'trx_addons_action_is_new_version_of_plugin', 'trx_addons_elm_add_spacer_default_value', 10, 2 );
	add_action( 'trx_addons_action_importer_import_end', 'trx_addons_elm_add_spacer_default_value' );
	/**
	 * Add a default value to all widgets 'Spacer' after update plugin ThemeREX Addons to the new version or after import demo data
	 * (because a new version of Elementor don't display the 'Spacer' widget with an empty value)
	 *
	 * @hooked trx_addons_action_is_new_version_of_plugin
	 * @hooked trx_addons_action_importer_import_end
	 * 
	 * @param string $new_version New version of the plugin.
	 * @param string $old_version Old version of the plugin.
	 */
	function trx_addons_elm_add_spacer_default_value( $new_version = '', $old_version = '' ) {
		if ( empty( $old_version ) ) {
			$old_version = get_option( 'trx_addons_version', '1.0' );
		}
		if ( version_compare( $old_version, '2.29.1', '<' ) || current_action() == 'trx_addons_action_importer_import_end' ) {
			global $wpdb;
			$rows = $wpdb->get_results( "SELECT post_id, meta_id, meta_value
											FROM {$wpdb->postmeta}
											WHERE meta_key='_elementor_data' && meta_value!=''"
										);
			if ( is_array( $rows ) && count( $rows ) > 0 ) {
				foreach ( $rows as $row ) {
					$data = json_decode( $row->meta_value, true );
					if ( trx_addons_elm_convert_spacer_default_value( $data ) ) {
						$wpdb->query( "UPDATE {$wpdb->postmeta} SET meta_value = '" . wp_slash( wp_json_encode( $data ) ) . "' WHERE meta_id = {$row->meta_id} LIMIT 1" );
					}
				}
			}
		}
	}
}

if ( ! function_exists( 'trx_addons_elm_convert_spacer_default_value' ) ) {
	/**
	 * Add a default value to all widgets 'Spacer' to the all elements in the Elementor data
	 * Attention! The parameter $elements passed by reference and modified inside this function!
	 * Return true if $elements is modified (converted) and needs to be saved
	 *
	 * @param array $elements  Array of elements. Passed by reference and modified inside this function!
	 * 
	 * @return boolean True if parameters was changed and needs to be saved
	 */
	function trx_addons_elm_convert_spacer_default_value( &$elements ) {
		$modified = false;
		$sc = array( 'spacer' );
		if ( is_array( $elements ) ) {
			foreach( $elements as $k => $elm ) {
				// Convert parameters
				if (   ! empty( $elm['widgetType'] )
					&&   in_array( $elm['widgetType'], $sc )
					&& ! empty( $elm['settings']['alter_height'] )
					&& ! empty( $elm['settings']['space'] )
					&&   empty( $elm['settings']['space']['size'] )
				) {
					unset( $elements[ $k ]['settings']['space'] );
					$modified = true;
				}
				// Process inner elements
				if ( ! empty( $elm['elements'] ) && is_array( $elm['elements'] ) ) {
					$modified = trx_addons_elm_convert_spacer_default_value( $elements[ $k ]['elements'] ) || $modified;
				}
			}
		}
		return $modified;
	}
}

// Demo data install
//----------------------------------------------------------------------------

// One-click import support
if ( is_admin() ) {
	require_once TRX_ADDONS_PLUGIN_DIR . TRX_ADDONS_PLUGIN_API . 'elementor/elementor-demo-importer.php';
}

// OCDI support
if ( is_admin() && trx_addons_exists_elementor() && trx_addons_exists_ocdi() ) {
	require_once TRX_ADDONS_PLUGIN_DIR . TRX_ADDONS_PLUGIN_API . 'elementor/elementor-demo-ocdi.php';
}
